#ifndef AHLGREN_LEVENSHTEIN
#define AHLGREN_LEVENSHTEIN

#include <string>
#include <vector>
#include <algorithm>
#include <iterator>
#include <exception>
#include "functor.h"

namespace lp {
	// Partial Order based on "subsequence distance"
	// "John" < "JaJohanEs"
	template <typename Itx, typename Ity>
	int subseq_dist(Itx xbeg, Itx xend, Ity ybeg, Ity yend)
	{
		int d = 0;
		for (;;) {
			if (xbeg == xend) return d + std::distance(ybeg,yend); // done
			if (ybeg == yend) return -1; // no match
			if (*xbeg == *ybeg) {
				++xbeg;
			} else {
				++d;
			}
			++ybeg;
		}
	}


	/*
	// Upper-Diagonal Matrix containing subseq_dist(i,i)
	// 0-1, 0-2, ...,              0-(i-1) <--- i-1 values
	//      1-2, 1-3, ...,         1-(i-1) <--- i-2 values
	//             ...
	//                         (i-2)-(i-1) <--- 1 value
	 (i,j) <=> vector[i]
	 0 <=> (0,1)
	 1 <=> (0,2)
	 ...
	 i-2  <=> (0,i-1)
	 i-1 <=> (1,2)
	 i <=> (1,3)

	*/


	// Algorithm to determine how dissimilar two examples are
	struct not_comparable : public std::exception {};
	int dissimilarity(const lp::Functor& e1, const lp::Functor& e2);

	// Compute difference as levenshtein distance
	template <typename Itx, typename Ity>
	int diff(Itx xbeg, Itx xend, int xlen, Ity ybeg, Ity yend, int ylen)
	{
		// Note: row has xlen+1 elements
		std::vector<int> prev_row, row;
		prev_row.reserve(xlen+1);
		for (int k = 0; k <= xlen; ++k) prev_row.push_back(k);
		row.resize(xlen+1);

		int col = 1;
		for ( ; ; ++col) {
			row[0] = col;
			int k = 1;
			for (auto i = xbeg; i != xend; ++i,++k) {
				if (*i == *ybeg) {
					row[k] = prev_row[k-1];
				} else {
					row[k] = 1 + std::min(row[k-1], std::min(prev_row[k], prev_row[k-1]));
				}
			}
			//// Print row
			//for (auto v : row) std::cout << v;
			//std::cout << "\n";
			if (++ybeg == yend) break;
			// Copy row to prev_row
			std::copy(row.begin(),row.end(),prev_row.begin());
		}

		return row.back();
	}


	// Compute difference as levenshtein distance
	template <typename Itx, typename Ity>
	int diff(Itx xbeg, Itx xend, Ity ybeg, Ity yend)
	{
		return diff(xbeg,xend,std::distance(xbeg,xend),ybeg,yend,std::distance(ybeg,yend));
	}


	int tdist(
		int xbeg, int xend,
		int ybeg, int yend,
		const std::vector<const lp::Functor*>& xl,
		const std::vector<const lp::Functor*>& yl);

	int tdist(const lp::Functor& f, const lp::Functor& g);

}


#endif


